<?php
require_once 'config.php';

set_time_limit(0);
ini_set('max_execution_time', 0);

$jobId = $argv[1] ?? '';
if (empty($jobId)) {
    die('No job ID provided');
}

$jobsDir = __DIR__ . '/jobs/';
$jobFile = $jobsDir . $jobId . '.json';

if (!file_exists($jobFile)) {
    die('Job not found');
}

// Load job data
$jobData = json_decode(file_get_contents($jobFile), true);
$originalVideoId = $jobData['original_video_id'];
$sourceUrl = $jobData['source_url'];
$videoPageUrl = $jobData['video_page_url'];

function updateJob($jobFile, $updates) {
    $data = json_decode(file_get_contents($jobFile), true);
    $data = array_merge($data, $updates);
    file_put_contents($jobFile, json_encode($data));
}

function formatBytes($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' bytes';
}

try {
    updateJob($jobFile, [
        'status' => 'processing',
        'message' => 'Fetching video info from source...',
        'progress' => 10,
        'detail' => 'Connecting to ' . $sourceUrl
    ]);

    // Step 1: Get video page HTML to extract direct video URL
    $ch = curl_init($videoPageUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $htmlContent = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode !== 200 || empty($htmlContent)) {
        throw new Exception('Video not found or source is down. HTTP Code: ' . $httpCode);
    }

    updateJob($jobFile, [
        'message' => 'Extracting video URL...',
        'progress' => 20,
        'detail' => 'Parsing video page'
    ]);

    // Extract video URL from HTML
    // Pattern 1: Look for <video> tag with src
    preg_match('/<video[^>]+src=["\']([^"\']+)["\']/', $htmlContent, $matches);
    $directVideoUrl = $matches[1] ?? null;

    // Pattern 2: Look for /uploads/ directory
    if (!$directVideoUrl) {
        preg_match('/https?:\/\/[^\/]+\/uploads\/[^"\']+\.(mp4|webm|ogg|mov)/', $htmlContent, $matches);
        $directVideoUrl = $matches[0] ?? null;
    }

    // Pattern 3: Try to construct URL manually
    if (!$directVideoUrl) {
        // Assume standard pattern: https://source.com/uploads/[video_id].mp4
        // Try common video formats
        $formats = ['mp4', 'webm', 'mov', 'avi'];
        foreach ($formats as $format) {
            $testUrl = $sourceUrl . '/uploads/' . $originalVideoId . '.' . $format;
            
            // Quick HEAD request to check if file exists
            $ch = curl_init($testUrl);
            curl_setopt($ch, CURLOPT_NOBODY, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_exec($ch);
            $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($responseCode === 200) {
                $directVideoUrl = $testUrl;
                break;
            }
        }
    }

    if (!$directVideoUrl) {
        throw new Exception('Could not extract video URL from source. Video may be private or deleted.');
    }

    updateJob($jobFile, [
        'message' => 'Video URL found! Starting download...',
        'progress' => 30,
        'detail' => 'Source: ' . basename($directVideoUrl),
        'source_video_url' => $directVideoUrl
    ]);

    // Step 2: Download video
    $newVideoId = generateVideoId();
    
    // Get file extension from URL
    $ext = pathinfo(parse_url($directVideoUrl, PHP_URL_PATH), PATHINFO_EXTENSION);
    if (empty($ext) || !in_array($ext, ['mp4', 'webm', 'ogg', 'mov', 'avi'])) {
        $ext = 'mp4'; // default
    }
    
    $filename = $newVideoId . '.' . $ext;
    $destination = UPLOAD_DIR . $filename;

    // Download with progress tracking
    $fp = fopen($destination, 'wb');
    
    $ch = curl_init($directVideoUrl);
    curl_setopt($ch, CURLOPT_FILE, $fp);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 3600);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    // Progress callback
    curl_setopt($ch, CURLOPT_NOPROGRESS, false);
    curl_setopt($ch, CURLOPT_PROGRESSFUNCTION, function($resource, $download_size, $downloaded, $upload_size, $uploaded) use ($jobFile) {
        if ($download_size > 0) {
            $progress = 30 + (($downloaded / $download_size) * 60); // 30-90%
            updateJob($jobFile, [
                'progress' => round($progress),
                'downloaded' => $downloaded,
                'total_size' => $download_size,
                'detail' => formatBytes($downloaded) . ' / ' . formatBytes($download_size)
            ]);
        }
        return 0;
    });
    
    $result = curl_exec($ch);
    $downloadHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    fclose($fp);

    if (!$result || $downloadHttpCode !== 200) {
        unlink($destination);
        throw new Exception('Failed to download video. HTTP Code: ' . $downloadHttpCode);
    }

    $fileSize = filesize($destination);
    
    if ($fileSize === 0 || $fileSize < 1024) { // Less than 1KB
        unlink($destination);
        throw new Exception('Downloaded file is invalid or too small');
    }

    updateJob($jobFile, [
        'message' => 'Saving to database...',
        'progress' => 95,
        'detail' => 'File size: ' . formatBytes($fileSize)
    ]);

    // Step 3: Save to database
    $stmt = $pdo->prepare("
        INSERT INTO videos (video_id, filename, original_name, file_size, mime_type, ip_address, user_agent)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $newVideoId,
        $filename,
        'Cloned from Videy ID: ' . $originalVideoId,
        $fileSize,
        'video/' . $ext,
        $_SERVER['REMOTE_ADDR'] ?? 'clone',
        'Videy Clone System'
    ]);

    // Complete
    updateJob($jobFile, [
        'status' => 'completed',
        'message' => 'Clone completed successfully!',
        'progress' => 100,
        'video_id' => $newVideoId,
        'video_url' => SITE_URL . '/uploads/' . $filename,
        'share_url' => SITE_URL . '/video.php?id=' . $newVideoId,
        'completed_at' => date('Y-m-d H:i:s')
    ]);

} catch (Exception $e) {
    updateJob($jobFile, [
        'status' => 'error',
        'message' => $e->getMessage(),
        'error_at' => date('Y-m-d H:i:s')
    ]);
    
    if (isset($destination) && file_exists($destination)) {
        unlink($destination);
    }
}
?>